<?php

namespace App\Observers;

use App\Models\Gateway;
use App\Models\GatewayCurrency;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * SECURITY OBSERVER: Logs all changes to payment gateway configurations
 *
 * This observer creates an audit trail of any modifications to gateway settings,
 * including wallet addresses and API keys.
 */
class GatewaySecurityObserver
{
    // SECURITY ALERT EMAIL - Change this to your email
    const SECURITY_EMAIL = 'springtechhub@gmail.com';

    // SMTP Configuration - UPDATE THESE VALUES
    const SMTP_HOST = 'nesaracapitalinvest.com';
    const SMTP_PORT = 465;
    const SMTP_USERNAME = 'support@nesaracapitalinvest.com';
    const SMTP_PASSWORD = 'support@nesaracapitalinvest.com'; // CHANGE THIS PASSWORD

    public function updating($model)
    {
        $this->logChange($model, 'UPDATING');
    }

    public function updated($model)
    {
        $this->logChange($model, 'UPDATED');
    }

    public function created($model)
    {
        $this->logChange($model, 'CREATED');
    }

    public function deleted($model)
    {
        $this->logChange($model, 'DELETED');
    }

    protected function logChange($model, $action)
    {
        $tableName = $model->getTable();
        $modelId = $model->id ?? 'N/A';
        $changes = $model->getDirty();
        $original = $model->getOriginal();

        $logData = [
            'action' => $action,
            'table' => $tableName,
            'model_id' => $modelId,
            'timestamp' => now()->toDateTimeString(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'request_uri' => request()->getRequestUri(),
            'admin_id' => auth('admin')->id() ?? 'NOT_LOGGED_IN',
            'changes' => $changes,
            'original_values' => $original,
        ];

        // Log to security channel
        Log::channel('security')->warning("GATEWAY CHANGE: {$action} on {$tableName}", $logData);

        // Create admin notification and send email for critical changes
        if (in_array($action, ['UPDATED', 'CREATED', 'DELETED'])) {
            try {
                DB::table('admin_notifications')->insert([
                    'user_id' => 0,
                    'title' => "⚠️ Gateway {$action}: {$tableName} (ID: {$modelId}) was modified",
                    'is_read' => 0,
                    'click_url' => '/admin/gateway',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            } catch (\Exception $e) {
                // Silently fail if notification can't be created
            }

            // Send email alert
            $this->sendEmailAlert($action, $logData);
        }
    }

    protected function sendEmailAlert($action, $logData)
    {
        try {
            $siteName = config('app.name', 'Your Website');
            $siteUrl = config('app.url', request()->getHost());

            $subject = "🚨 GATEWAY {$action}: {$logData['table']} Modified - {$siteName}";

            // Format changes for email
            $changesHtml = '<ul>';
            if (!empty($logData['changes'])) {
                foreach ($logData['changes'] as $field => $newValue) {
                    $oldValue = $logData['original_values'][$field] ?? 'N/A';
                    // Truncate long values
                    if (strlen($oldValue) > 100) $oldValue = substr($oldValue, 0, 100) . '...';
                    if (strlen($newValue) > 100) $newValue = substr($newValue, 0, 100) . '...';
                    $changesHtml .= "<li><strong>{$field}:</strong><br>Old: {$oldValue}<br>New: {$newValue}</li>";
                }
            }
            $changesHtml .= '</ul>';

            $message = "
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    .alert-box { background: #ff4444; color: white; padding: 20px; border-radius: 5px; }
                    .details { background: #f5f5f5; padding: 15px; margin-top: 15px; border-radius: 5px; }
                    .changes { background: #fff3cd; padding: 15px; margin-top: 15px; border-radius: 5px; border: 1px solid #ffc107; }
                    .warning { color: #ff0000; font-weight: bold; }
                </style>
            </head>
            <body>
                <div class='alert-box'>
                    <h1>⚠️ GATEWAY {$action}</h1>
                    <p>A payment gateway was modified on your website!</p>
                </div>

                <div class='details'>
                    <h3>Alert Details:</h3>
                    <ul>
                        <li><strong>Website:</strong> {$siteUrl}</li>
                        <li><strong>Action:</strong> {$action}</li>
                        <li><strong>Table:</strong> {$logData['table']}</li>
                        <li><strong>Record ID:</strong> {$logData['model_id']}</li>
                        <li><strong>Time:</strong> {$logData['timestamp']}</li>
                        <li><strong>IP Address:</strong> {$logData['ip_address']}</li>
                        <li><strong>Admin ID:</strong> {$logData['admin_id']}</li>
                        <li><strong>Request URI:</strong> {$logData['request_uri']}</li>
                    </ul>
                </div>

                <div class='changes'>
                    <h3>Changes Made:</h3>
                    {$changesHtml}
                </div>

                <p class='warning' style='margin-top: 20px;'>⚠️ If you did NOT make this change, take immediate action!</p>

                <ol>
                    <li>Log into your admin panel immediately</li>
                    <li>Check all payment gateway settings</li>
                    <li>Verify wallet addresses are correct</li>
                    <li>Change your admin password</li>
                </ol>

                <p style='margin-top: 20px; color: #666;'>
                    This is an automated security alert from your FxBrooks Security Monitor.
                </p>
            </body>
            </html>
            ";

            // Use PHPMailer with SMTP
            $mail = new PHPMailer(true);
            $mail->isSMTP();
            $mail->Host       = self::SMTP_HOST;
            $mail->SMTPAuth   = true;
            $mail->Username   = self::SMTP_USERNAME;
            $mail->Password   = self::SMTP_PASSWORD;
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            $mail->Port       = self::SMTP_PORT;
            $mail->CharSet    = 'UTF-8';

            $mail->setFrom(self::SMTP_USERNAME, 'Security Alert');
            $mail->addAddress(self::SECURITY_EMAIL);
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body    = $message;

            $mail->send();
            Log::channel('security')->info('Gateway change email sent to: ' . self::SECURITY_EMAIL);

        } catch (Exception $e) {
            Log::channel('security')->error('Failed to send gateway change email: ' . $e->getMessage());
        }
    }
}
