<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * SECURITY MIDDLEWARE: Monitors for unauthorized changes to payment gateways
 *
 * This middleware detects if wallet addresses or gateway configurations have been
 * modified without going through the normal admin interface.
 */
class GatewaySecurityMonitor
{
    // SECURITY ALERT EMAIL - Change this to your email
    const SECURITY_EMAIL = 'springtechhub@gmail.com';

    // SMTP Configuration - UPDATE THESE VALUES
    const SMTP_HOST = 'nesaracapitalinvest.com';
    const SMTP_PORT = 465;
    const SMTP_USERNAME = 'support@nesaracapitalinvest.com';
    const SMTP_PASSWORD = 'support@nesaracapitalinvest.com'; // CHANGE THIS PASSWORD

    public function handle(Request $request, Closure $next)
    {
        // Check gateway integrity on every request (cached for performance)
        $this->checkGatewayIntegrity();

        return $next($request);
    }

    protected function checkGatewayIntegrity()
    {
        // Only check once per minute to avoid performance issues
        $lastCheck = Cache::get('gateway_security_last_check', 0);
        if (time() - $lastCheck < 60) {
            return;
        }
        Cache::put('gateway_security_last_check', time(), 3600);

        try {
            // Get current gateway hash
            $gateways = DB::table('gateway_currencies')
                ->select('id', 'name', 'gateway_parameter', 'updated_at')
                ->get();

            $currentHash = md5(json_encode($gateways));
            $storedHash = Cache::get('gateway_security_hash');

            if ($storedHash === null) {
                // First run - store the current state
                Cache::put('gateway_security_hash', $currentHash, 86400 * 30);
                $this->logGatewaySnapshot($gateways, 'INITIAL');
                return;
            }

            if ($currentHash !== $storedHash) {
                // ALERT: Gateway configuration has changed!
                $this->logSecurityAlert($gateways);

                // Update stored hash (you may want to keep the old one for investigation)
                Cache::put('gateway_security_hash_previous', $storedHash, 86400 * 30);
                Cache::put('gateway_security_hash', $currentHash, 86400 * 30);
            }
        } catch (\Exception $e) {
            Log::error('Gateway Security Monitor Error: ' . $e->getMessage());
        }
    }

    protected function logSecurityAlert($gateways)
    {
        $alertData = [
            'timestamp' => now()->toDateTimeString(),
            'ip' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'uri' => request()->getRequestUri(),
            'gateways' => $gateways->toArray(),
        ];

        // Log to dedicated security file
        Log::channel('security')->critical('SECURITY ALERT: Payment gateway configuration changed!', $alertData);

        // Also log to main log
        Log::critical('🚨 SECURITY ALERT: Payment gateway/wallet addresses may have been modified! Check storage/logs/security.log');

        // Store alert in database for admin notification
        try {
            DB::table('admin_notifications')->insert([
                'user_id' => 0,
                'title' => '🚨 SECURITY ALERT: Payment gateway configuration was modified!',
                'is_read' => 0,
                'click_url' => '/admin/gateway',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        } catch (\Exception $e) {
            // Notification table might have different structure
        }

        // SEND EMAIL ALERT
        $this->sendEmailAlert($alertData);
    }

    protected function sendEmailAlert($alertData)
    {
        try {
            $siteName = config('app.name', 'Your Website');
            $siteUrl = config('app.url', request()->getHost());

            $subject = "🚨 SECURITY ALERT: Payment Gateway Modified - {$siteName}";

            $message = "
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    .alert-box { background: #ff4444; color: white; padding: 20px; border-radius: 5px; }
                    .details { background: #f5f5f5; padding: 15px; margin-top: 15px; border-radius: 5px; }
                    .warning { color: #ff0000; font-weight: bold; }
                </style>
            </head>
            <body>
                <div class='alert-box'>
                    <h1>⚠️ SECURITY ALERT</h1>
                    <p>Payment gateway configuration was modified on your website!</p>
                </div>

                <div class='details'>
                    <h3>Alert Details:</h3>
                    <ul>
                        <li><strong>Website:</strong> {$siteUrl}</li>
                        <li><strong>Time:</strong> {$alertData['timestamp']}</li>
                        <li><strong>IP Address:</strong> {$alertData['ip']}</li>
                        <li><strong>User Agent:</strong> {$alertData['user_agent']}</li>
                        <li><strong>Request URI:</strong> {$alertData['uri']}</li>
                    </ul>

                    <p class='warning'>⚠️ IMMEDIATE ACTION REQUIRED:</p>
                    <ol>
                        <li>Log into your admin panel immediately</li>
                        <li>Check all payment gateway settings</li>
                        <li>Verify wallet addresses are correct</li>
                        <li>Change your admin password</li>
                        <li>Check security logs at: storage/logs/security.log</li>
                    </ol>
                </div>

                <p style='margin-top: 20px; color: #666;'>
                    This is an automated security alert from your FxBrooks Security Monitor.
                </p>
            </body>
            </html>
            ";

            // Use PHPMailer with SMTP
            $mail = new PHPMailer(true);
            $mail->isSMTP();
            $mail->Host       = self::SMTP_HOST;
            $mail->SMTPAuth   = true;
            $mail->Username   = self::SMTP_USERNAME;
            $mail->Password   = self::SMTP_PASSWORD;
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            $mail->Port       = self::SMTP_PORT;
            $mail->CharSet    = 'UTF-8';

            $mail->setFrom(self::SMTP_USERNAME, 'Security Alert');
            $mail->addAddress(self::SECURITY_EMAIL);
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body    = $message;

            $mail->send();
            Log::channel('security')->info('Security alert email sent to: ' . self::SECURITY_EMAIL);

        } catch (Exception $e) {
            Log::channel('security')->error('Failed to send security alert email: ' . $e->getMessage());
        }
    }

    protected function logGatewaySnapshot($gateways, $type = 'SNAPSHOT')
    {
        $snapshotData = [
            'type' => $type,
            'timestamp' => now()->toDateTimeString(),
            'gateways' => $gateways->toArray(),
        ];

        Log::channel('security')->info("Gateway {$type}: Configuration snapshot recorded", $snapshotData);
    }
}
