<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * SECURITY MIDDLEWARE: Monitors for unauthorized changes to critical files
 *
 * This middleware detects if deposit pages, payment files, or wallet-related
 * files have been modified.
 */
class FileIntegrityMonitor
{
    // SECURITY ALERT EMAIL
    const SECURITY_EMAIL = 'springtechhub@gmail.com';

    // SMTP Configuration - UPDATE THESE VALUES
    const SMTP_HOST = 'nesaracapitalinvest.com';
    const SMTP_PORT = 465;
    const SMTP_USERNAME = 'support@nesaracapitalinvest.com';
    const SMTP_PASSWORD = 'support@nesaracapitalinvest.com'; // CHANGE THIS PASSWORD

    // Critical files/directories to monitor
    protected $criticalPaths = [
        // All payment blade files
        'resources/views/templates/invester/user/payment',
        'resources/views/templates/bit_gold/user/payment',
        'resources/views/templates/neo_dark/user/payment',

        // Gateway controllers
        'app/Http/Controllers/Gateway',

        // Core payment controller
        'app/Http/Controllers/Gateway/PaymentController.php',

        // Helpers (contains wallet functions)
        'app/Http/Helpers/helpers.php',

        // Curl requests (external communication)
        'app/Lib/CurlRequest.php',
    ];

    public function handle(Request $request, Closure $next)
    {
        $this->checkFileIntegrity();
        return $next($request);
    }

    protected function checkFileIntegrity()
    {
        // Only check once every 5 minutes
        $lastCheck = Cache::get('file_integrity_last_check', 0);
        if (time() - $lastCheck < 300) {
            return;
        }
        Cache::put('file_integrity_last_check', time(), 3600);

        try {
            $currentHashes = $this->getFileHashes();
            $storedHashes = Cache::get('file_integrity_hashes');

            if ($storedHashes === null) {
                // First run - store current state
                Cache::put('file_integrity_hashes', $currentHashes, 86400 * 30);
                Log::channel('security')->info('File Integrity Monitor: Initial file hashes recorded', [
                    'file_count' => count($currentHashes),
                    'timestamp' => now()->toDateTimeString()
                ]);
                return;
            }

            // Compare hashes
            $changes = $this->compareHashes($storedHashes, $currentHashes);

            if (!empty($changes['modified']) || !empty($changes['added']) || !empty($changes['deleted'])) {
                $this->sendFileChangeAlert($changes);

                // Update stored hashes
                Cache::put('file_integrity_hashes_previous', $storedHashes, 86400 * 30);
                Cache::put('file_integrity_hashes', $currentHashes, 86400 * 30);
            }

        } catch (\Exception $e) {
            Log::error('File Integrity Monitor Error: ' . $e->getMessage());
        }
    }

    protected function getFileHashes()
    {
        $hashes = [];
        $basePath = base_path();

        foreach ($this->criticalPaths as $relativePath) {
            $fullPath = $basePath . '/' . $relativePath;

            if (is_file($fullPath)) {
                $hashes[$relativePath] = [
                    'hash' => md5_file($fullPath),
                    'modified' => filemtime($fullPath),
                    'size' => filesize($fullPath)
                ];
            } elseif (is_dir($fullPath)) {
                // Get all PHP and blade files in directory
                $files = File::allFiles($fullPath);
                foreach ($files as $file) {
                    if (in_array($file->getExtension(), ['php', 'blade'])) {
                        $relative = str_replace($basePath . '/', '', $file->getPathname());
                        // Normalize path separators
                        $relative = str_replace('\\', '/', $relative);
                        $hashes[$relative] = [
                            'hash' => md5_file($file->getPathname()),
                            'modified' => filemtime($file->getPathname()),
                            'size' => filesize($file->getPathname())
                        ];
                    }
                }
            }
        }

        return $hashes;
    }

    protected function compareHashes($old, $new)
    {
        $changes = [
            'modified' => [],
            'added' => [],
            'deleted' => []
        ];

        // Check for modified and deleted files
        foreach ($old as $path => $data) {
            if (!isset($new[$path])) {
                $changes['deleted'][] = $path;
            } elseif ($new[$path]['hash'] !== $data['hash']) {
                $changes['modified'][] = [
                    'path' => $path,
                    'old_hash' => $data['hash'],
                    'new_hash' => $new[$path]['hash'],
                    'old_modified' => date('Y-m-d H:i:s', $data['modified']),
                    'new_modified' => date('Y-m-d H:i:s', $new[$path]['modified']),
                    'old_size' => $data['size'],
                    'new_size' => $new[$path]['size']
                ];
            }
        }

        // Check for new files
        foreach ($new as $path => $data) {
            if (!isset($old[$path])) {
                $changes['added'][] = $path;
            }
        }

        return $changes;
    }

    protected function sendFileChangeAlert($changes)
    {
        $alertData = [
            'timestamp' => now()->toDateTimeString(),
            'ip' => request()->ip(),
            'changes' => $changes
        ];

        // Log the alert
        Log::channel('security')->critical('🚨 FILE INTEGRITY ALERT: Critical files were modified!', $alertData);

        // Admin notification
        try {
            DB::table('admin_notifications')->insert([
                'user_id' => 0,
                'title' => '🚨 CRITICAL: Payment/deposit files were modified!',
                'is_read' => 0,
                'click_url' => '/admin/dashboard',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        } catch (\Exception $e) {}

        // Send email
        $this->sendEmailAlert($changes);
    }

    protected function sendEmailAlert($changes)
    {
        try {
            $siteName = config('app.name', 'Your Website');
            $siteUrl = config('app.url', request()->getHost());

            $subject = "🚨 CRITICAL ALERT: Payment Files Modified - {$siteName}";

            // Format changes for email
            $modifiedHtml = '';
            if (!empty($changes['modified'])) {
                $modifiedHtml = '<h4 style="color: #ff0000;">Modified Files:</h4><ul>';
                foreach ($changes['modified'] as $file) {
                    $modifiedHtml .= "<li><strong>{$file['path']}</strong><br>";
                    $modifiedHtml .= "Changed: {$file['old_modified']} → {$file['new_modified']}<br>";
                    $modifiedHtml .= "Size: {$file['old_size']} → {$file['new_size']} bytes</li>";
                }
                $modifiedHtml .= '</ul>';
            }

            $addedHtml = '';
            if (!empty($changes['added'])) {
                $addedHtml = '<h4 style="color: #ff6600;">New Files Added:</h4><ul>';
                foreach ($changes['added'] as $file) {
                    $addedHtml .= "<li>{$file}</li>";
                }
                $addedHtml .= '</ul>';
            }

            $deletedHtml = '';
            if (!empty($changes['deleted'])) {
                $deletedHtml = '<h4 style="color: #990000;">Files Deleted:</h4><ul>';
                foreach ($changes['deleted'] as $file) {
                    $deletedHtml .= "<li>{$file}</li>";
                }
                $deletedHtml .= '</ul>';
            }

            $message = "
            <html>
            <head>
                <style>
                    body { font-family: Arial, sans-serif; }
                    .alert-box { background: #cc0000; color: white; padding: 20px; border-radius: 5px; }
                    .details { background: #f5f5f5; padding: 15px; margin-top: 15px; border-radius: 5px; }
                    .files { background: #fff3cd; padding: 15px; margin-top: 15px; border-radius: 5px; border: 1px solid #ff0000; }
                    .warning { color: #ff0000; font-weight: bold; font-size: 18px; }
                </style>
            </head>
            <body>
                <div class='alert-box'>
                    <h1>🚨 CRITICAL FILE CHANGE ALERT</h1>
                    <p>Payment and deposit files were modified on your website!</p>
                    <p><strong>This is likely a hack attempt!</strong></p>
                </div>

                <div class='details'>
                    <h3>Alert Details:</h3>
                    <ul>
                        <li><strong>Website:</strong> {$siteUrl}</li>
                        <li><strong>Time:</strong> " . now()->toDateTimeString() . "</li>
                        <li><strong>Detection IP:</strong> " . request()->ip() . "</li>
                    </ul>
                </div>

                <div class='files'>
                    <h3>File Changes Detected:</h3>
                    {$modifiedHtml}
                    {$addedHtml}
                    {$deletedHtml}
                </div>

                <p class='warning' style='margin-top: 20px;'>⚠️ IMMEDIATE ACTION REQUIRED:</p>
                <ol>
                    <li><strong>DO NOT</strong> accept any deposits until you verify files</li>
                    <li>Check the modified files for malicious wallet addresses</li>
                    <li>Restore files from a known good backup</li>
                    <li>Change ALL admin passwords</li>
                    <li>Check your FTP/hosting access logs</li>
                    <li>Consider taking the site offline until fixed</li>
                </ol>

                <p style='margin-top: 20px; color: #666;'>
                    This is an automated security alert from your FxBrooks Security Monitor.
                </p>
            </body>
            </html>
            ";

            // Use PHPMailer with SMTP
            $mail = new PHPMailer(true);
            $mail->isSMTP();
            $mail->Host       = self::SMTP_HOST;
            $mail->SMTPAuth   = true;
            $mail->Username   = self::SMTP_USERNAME;
            $mail->Password   = self::SMTP_PASSWORD;
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            $mail->Port       = self::SMTP_PORT;
            $mail->CharSet    = 'UTF-8';

            $mail->setFrom(self::SMTP_USERNAME, 'Security Alert');
            $mail->addAddress(self::SECURITY_EMAIL);
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body    = $message;

            $mail->send();
            Log::channel('security')->info('File change alert email sent to: ' . self::SECURITY_EMAIL);

        } catch (Exception $e) {
            Log::channel('security')->error('Failed to send file change alert email: ' . $e->getMessage());
        }
    }
}
